(* ::Subsubsection:: *)
(*MaksutovCassegrain package*)

(* ::Text:: *)
(*This impements the Maksutov Cassegrain telescope optical system*)

(* ::Text:: *)
(*Chapter*)

(* ::Text:: *)
(*This file should be in the following path*)

(* ::Text:: *)
(*FileNameJoin[{$UserBaseDirectory, "Applications/GeometricOptics/MaksutovCassegrain"}]*)

BeginPackage["GeometricOptics`Packages`MaksutovCassegrain`", {"GeometricOptics`"}]

Options[MaksutovCassegrain] = {OutputType->"Report", OutputLevel->"Full"};
(* 	OutputType can be 
					"Report", generates a new report (notebook) for each call to the function
					"Print", prints results in the current position of the evaluating notebook
					"Basic", gives the list of results
					"Values" or any other directive, gives the list of values (no names of variables)
	OutputLevel can be
					"Minimal", gives only output values, those calculated by the function
					"Full", gives the list of input parameters and output values
*)
MaksutovCassegrain::usage="MaksutovCassegrain calculates parameters for a Maksutov Camera optical system.";

MaksutovCassegrainInputPanel::usage = "MaksutovCassegrainInputPanel gives the GUI panel to work with MaksutovCassegrain";

$MaksutovCassegrainInputVariables = {"f1", "ft", "em", "\[Delta]", "diam", "\[Theta]"};

Begin["`Private`"]

MaksutovCassegrain[{f1_, ft_, em_, delta_, diam_, theta_}, opts___]:= MaksutovCassegrain[f1, ft, em, delta, diam, theta, opts];

MaksutovCassegrain[f1_, ft_, em_, delta_, diam_, theta_, OptionsPattern[]] := 
 Module[{\[Alpha], M, \[Beta], \[Gamma], r, \[CapitalDelta], Ni, s, rad1, thick1, ind, costasf, outputs, inPanel, outPanel},
 
		If[ArgumentsQ["MaksutovCassegrain", {$MaksutovCassegrainInputVariables, {f1, ft, em, delta, diam, theta}}],
			AppendTo[$ExamplesStack, 
					 <|"PackageID" -> 10, "PackageName" -> "MaksutovCassegrain", "Arguments" ->  <|"f1" -> f1, "ft" -> ft, "em" -> em, "\[Delta]" -> delta, "diam" -> diam, "\[Theta]" -> theta|>|>];
			$ExamplesStack = DeleteExampleDuplicates[$ExamplesStack];

			\[Alpha] = em/f1;
			M = ft/ f1;
			\[Beta] = (M - \[Alpha])/(M + 1);
			\[Gamma] = (2 M (1 + \[Alpha]))/(1 - M^2);
			r = diam/2;
			\[CapitalDelta] = 0.1 diam;
			Ni = 1.5187228;
			s = \[CapitalDelta] (Ni^2 - 1)/(0.97 Ni^2);
			(* Correction of curvature radii due to the finite focal of corrector *)
			rad1 = {R1, R1 - s, -2 f1, \[Gamma] f1};
			thick1 = {\[CapitalDelta], delta f1, -\[Beta] f1};
			ind = {{1, Ni, 1, -1, 1}};
			costasf = {0, 0, 0, 0};
			TotalAberrations[rad1, thick1, ind, costasf, diam/2, 0, 0, -Infinity, x, theta, {\[Lambda]}];
			r1 = Min[Replace[R1, Solve[Numerator[Together[GOSphericalCoefficient]] == 0, R1, Reals]]];
			(* Evaluation of aberrations *)
			rad1 = {r1, r1 - s, -2 f1, \[Gamma] f1};
			thick1 = {\[CapitalDelta], delta f1, -\[Beta] f1};
			TotalAberrations[rad1, thick1, ind, costasf, diam/2, 0, 0, -Infinity, x, theta, {\[Lambda]}];
			
			outputs = { {"Anterior curvature radius of corrector", r1},
						{"Posterior curvature radius of corrector", r1 - s},
						{"Distance between the corrector and mirror", delta f1},
						{"Distance between the primary and secondary mirror", N[-\[Beta] f1]},
						{"Radius of the secondary mirror", N[\[Gamma] f1]},
						{"Transversal spherical aberration", GOaberration[[1]]},
						{"Total sagittal coma", GOcoma[[1]]},
						{"Total astigmatism", GOastigmatism[[1]]},
						{"Focal length", GOfocalLength[[1]]},
						{"Image height", GOimageHeight[[1, GOn]]}};

			(* defines the two panels, input parameters and output values *)
			inPanel = Grid[{{"Focal length of the primary mirror" , "f1", f1},
							{"Total focal length", "ft", ft},
							{"Back distance", "em", em},
							{"Fraction of distance corrector-primary in focal unity", "delta", delta},
							{"Diameter of primary mirror", "diam", diam},
							{"Field angle  in degree", "theta", theta}},
							Alignment -> {{Left, Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"InputParameterBottom"}];
  
			outPanel = Grid[outputs, 
							Alignment -> {{Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"OutputValueBottom"}];
				
			(* generates the type of output required *)
			Switch[OptionValue[OutputType],
					"Report",
					GenerateDocument[TemplateApply[$ReportTemplate, 
										Join[<|	"title" -> $GeometricOpticsPackagesList[SelectFirst[#PackageName == "MaksutovCassegrain" &], "Description"], 
												"date" -> DateString[], 
												"function" -> "MaksutovCassegrain", 
												"outputlevel" -> OptionValue[OutputLevel],
												"inPanel" -> inPanel, 
												"outPanel" -> outPanel |>]]];,

					"Print",
					CellPrint[TextCell[TemplateApply[$PrintTemplate, 
											Join[<|	"title" -> $GeometricOpticsPackagesList[SelectFirst[#PackageName == "MaksutovCassegrain" &], "Description"], 
													"date" -> DateString[], 
													"function" -> "MaksutovCassegrain", 
													"outputlevel" -> OptionValue[OutputLevel],
													"inPanel" -> inPanel, 
													"outPanel" -> outPanel |>]], "Text"]];,
					"Basic",
					CellPrint[TextCell[
					TemplateApply[$BasicTemplate, 
									Join[<| "outputlevel" -> OptionValue[OutputLevel],
											"inputs" -> {{"f1" , f1},
														 {"ft", ft},
														 {"em", em},
														 {"\[Delta]" , delta},
														 {"diam", diam},
														 {"\[Theta]", theta}},
											"outputs" -> outputs |>]], "Output"]],
					_,
					CellPrint[TextCell[DeleteCases[outputs[[All, 2]], Alternatives["", Style[___]]], "Output"]]],

							
			(* Arguments are not correct *)
			MessageDialog["MaksutovCassegrain not executed, the number or the type of the arguments may be incorrect.", WindowTitle->"Warning: example not generated"];]];

MaksutovCassegrainInputPanel[]:=
DynamicModule[{outputtype = "Report", package = "MaksutovCassegrain"},
	examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
						Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
    examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], Alignment -> Center, Spacings -> {1, 1}, Dividers -> Center], FrameStyle -> LightGray] &, examplesAll];
	example = "Browse...";
	Panel[Column[{	DynamicWrapper[Style[NameFromPackageName[package] <> " Input Panel", "Subsection"], 
									If[NumericQ[example], {f1, ft, em, delta, diam, theta} = ReplaceAll[$MaksutovCassegrainInputVariables, examplesAll[[example]]]]],
					Style["Insert values for each argument, then use Evaluate to run "<>package<>" function", "Text"],
					Grid[{{"Focal length of the primary mirror" , "f1", Tooltip[InputField[Dynamic[f1], Alignment -> Center], "Insert the value of focal length of the primary mirror"]},
						  {"Total focal length", "ft", Tooltip[InputField[Dynamic[ft], Alignment -> Center], "Insert the total focal length"]},
						  {"Back distance", "em", Tooltip[InputField[Dynamic[em], Alignment -> Center], "Insert the back distance"]},
						  {"Fraction of distance corrector-primary in focal unity", "\[Delta]", Tooltip[InputField[Dynamic[delta], Alignment -> Center], "Insert the \[Delta] coefficient"]},
						  {"Diameter of the primary mirror", "diam", Tooltip[InputField[Dynamic[diam], Alignment -> Center], "Input the value of diam"]},
						  {"Field angle in degrees", "\[Theta]", Tooltip[InputField[Dynamic[theta], Alignment -> Center], "Insert the value of \[Theta]"]}},
						Spacings -> {1, 0},
						Alignment -> {{Left, Left, Right}, Center}, 
						Dividers -> Center, 
						FrameStyle -> LightGray],
					OpenerView[{"Load an example from the archives (current session and saved DB)",
								Row[{Dynamic@PopupMenu[Dynamic[example], Thread[Rule[Range[Length[examples]], examples]], If[examples === {}, "No example saved", "Browse..."], FrameMargins -> 3, Alignment -> Center],
									 Spacer[5],
									 Button["Update list", (examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
																				Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
															examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], 
																						Alignment -> Center, 
																						Spacings -> {1, 1}, 
																						Dividers -> Center], 
																						FrameStyle -> LightGray]&, examplesAll];
															example = "Browse..."), Method -> "Queued"]}]}, 
								Alignment -> Center, 
								Spacings -> 1.5],
					Row[{"Define the type of output to generate", 
						 Spacer[5],
						 RadioButtonBar[Dynamic[outputtype], {"Report" -> Tooltip["Report", "Generates a new notebook reporting a summary of the calculation"], 
															  "Print" -> Tooltip["Print", "Print the table of the calculation done inside the current notebook"], 
															  "Basic" -> Tooltip["Basic", "Generate a list of computed output with label"], 
															  "Values" -> Tooltip["Values", "Return only the list of output values"]}]}],
					Row[{Button["Evaluate", ToExpression[package][Apply[Sequence, {f1, ft, em, delta, diam, theta}], OutputType -> outputtype], Method -> "Queued"],
						 Button["Clear all", Map[Clear, Unevaluated[{f1, ft, em, delta, diam, theta}]]]}]}, 
				Spacings -> 2, 
				Alignment -> Center],
	BaseStyle -> {InputFieldBoxOptions -> {FieldSize -> {15, 1}}}]];
			
  
  End[]
  EndPackage[]